/*:ja
 * @target MZ
 * @plugindesc 味方アクターの自動戦闘における行動パターンを詳細に設定します。
 * @author 柊 七紀, Gemini
 * @help
 * ActorAIController.js
 *
 * このプラグインは、自動戦闘に設定されたアクターの行動AIを、
 * 敵キャラクターのように条件と優先度（レーティング）に基づいて
 * 設定できるようにします。
 *
 * --- 使い方 ---
 * アクターのメモ欄に、特定の形式で行動パターンを記述します。
 * 記述がないアクターは、通常の自動戦闘AIで行動します。
 *
 * ■ 基本的な書式
 * <AutoAction: スキルID, レーティング, 条件タイプ, 条件パラメータ1, [条件パラメータ2]>
 *
 * ・スキルID: 使用させたいスキルのID。
 * ・レーティング: 行動の優先度。数値が大きいほど優先されます（1～9）。
 * 　同じ優先度のスキルが複数ある場合、その中からランダムで選択されます。
 * ・条件タイプ: スキルを使用するための条件を指定します。
 * 　（Turn, HP, Switch, State, Level, EquipW, EquipA のいずれか。
 * 指定しない場合は常に条件を満たします）
 * ・条件パラメータ1, 2: 条件タイプによって必要な値。
 *
 * --- 条件の種類とパラメータ ---
 *
 * 1. Turn (ターン数)
 * 指定したターン数にのみ、その行動が選択肢に含まれるようになります。
 * 書式: <AutoAction: スキルID, レーティング, Turn, A, B>
 * ・A: ターン数を指定します。
 * ・B: 繰り返す間隔を指定します。「A + B * X」ターン目（Xは0以上）
 * が条件になります。Bを0にすると、Aターン目のみが条件になります。
 * 例）
 * <AutoAction: 10, 8, Turn, 1, 0>  // 1ターン目にスキルID10を優先度8で使用
 * <AutoAction: 12, 9, Turn, 2, 3>  // 2, 5, 8, 11...ターン目にスキルID12を優先度9で使用
 *
 * 2. HP (HPの割合)
 * 自身のHPが指定した割合の範囲内にある場合のみ、行動が選択肢に含まれます。
 * 書式: <AutoAction: スキルID, レーティング, HP, A, B>
 * ・A: HP割合の下限 (%)
 * ・B: HP割合の上限 (%)
 * 例）
 * <AutoAction: 25, 9, HP, 0, 50>  // HPが50%以下の時に回復スキル(ID:25)を優先度9で使用
 * <AutoAction: 30, 7, HP, 80, 100> // HPが80%以上の時に強化スキル(ID:30)を優先度7で使用
 *
 * 3. Switch (スイッチ)
 * 指定したスイッチがONの場合のみ、行動が選択肢に含まれるようになります。
 * 書式: <AutoAction: スキルID, レーティング, Switch, スイッチID>
 * 例）
 * <AutoAction: 55, 9, Switch, 10> // スイッチ10番がONの時にスキルID55を優先度9で使用
 *
 * 4. State (ステート)
 * 指定したステートにかかっている場合のみ、行動が選択肢に含まれます。
 * 書式: <AutoAction: スキルID, レーティング, State, ステートID>
 * 例）
 * <AutoAction: 60, 8, State, 5> // ステートID5番(例:毒)にかかっている時にスキルID60を使用
 *
 * 5. Level (レベル)
 * 自身のレベルが指定した値以上の場合のみ、行動が選択肢に含まれます。
 * 書式: <AutoAction: スキルID, レーティング, Level, レベル>
 * 例）
 * <AutoAction: 70, 7, Level, 20> // レベル20以上の時にスキルID70を使用
 *
 * 6. EquipW (武器装備)
 * 指定したIDの武器を装備している場合のみ、行動が選択肢に含まれます。
 * 書式: <AutoAction: スキルID, レーティング, EquipW, 武器ID>
 * 例）
 * <AutoAction: 80, 8, EquipW, 3> // 武器ID3番を装備している時にスキルID80を使用
 *
 * 7. EquipA (防具装備)
 * 指定したIDの防具を装備している場合のみ、行動が選択肢に含まれます。
 * 書式: <AutoAction: スキルID, レーティング, EquipA, 防具ID>
 * 例）
 * <AutoAction: 90, 7, EquipA, 5> // 防具ID5番を装備している時にスキルID90を使用
 *
 * --- 基本行動について ---
 * 通常攻撃や防御も、スキルとしてIDが割り振られています。
 * ・通常攻撃: スキルID 1
 * ・防御: スキルID 2
 *
 * これらもAutoActionに設定可能です。
 * 例）
 * <AutoAction: 2, 3, HP, 0, 30> // HPが30%以下なら優先度3で防御する
 *
 * --- 注意事項 ---
 * ・条件を満たすスキルが一つもない場合、そのアクターは「防御」します。
 * ・複数の行動を記述することで、複雑なAIを構築できます。
 * ・このプラグインは「自動戦闘」特性を持つアクターにのみ適用されます。
 * 作戦コマンドで「戦う」を選んだ場合は適用されません。
 * ・条件パラメータには有効なIDや数値を指定してください。
 */

(() => {
    "use strict";

    const PLUGIN_NAME = "ActorAIController"; // プラグイン名を修正 (任意)

    //-----------------------------------------------------------------------------
    // DataManager
    // アクターのメモ欄からAI行動パターンを読み込む
    //-----------------------------------------------------------------------------

    const _DataManager_onLoad = DataManager.onLoad;
    DataManager.onLoad = function(object) {
        _DataManager_onLoad.call(this, object);
        // $dataActors の読み込み完了時にパース処理を実行
        if (object === $dataActors) {
            for (let i = 1; i < object.length; i++) {
                const actor = object[i];
                if (actor) {
                    this.parseActorAutoBattleActions(actor);
                }
            }
        }
    };

    DataManager.parseActorAutoBattleActions = function(actor) {
        actor.autoBattleActions = [];
        const note = actor.note;
        // 正規表現を少し修正して、より多くの空白に対応
        const reg = /<AutoAction:\s*([^>]+)>/gi;
        let match;
        while ((match = reg.exec(note)) !== null) {
            const params = match[1].split(',').map(s => s.trim());
            if (params.length < 2) continue; // 最低でも スキルID と レーティング は必要

            const action = {};
            action.skillId = parseInt(params[0], 10);
            action.rating = parseInt(params[1], 10) || 5;
            // 条件タイプがない場合も考慮
            action.conditionType = params[2] ? params[2].toUpperCase() : 'NONE';
            action.conditionParam1 = params[3] ? parseInt(params[3], 10) : 0;
            action.conditionParam2 = params[4] ? parseInt(params[4], 10) : 0;

            // パラメータが数値として解釈できるか簡易チェック
            if (isNaN(action.skillId) || isNaN(action.rating)) {
                 console.warn(`[${PLUGIN_NAME}] Actor ${actor.id} のメモ欄で無効な AutoAction タグが見つかりました: ${match[0]}`);
                 continue;
            }

            actor.autoBattleActions.push(action);
        }
    };

    //-----------------------------------------------------------------------------
    // Game_Actor
    // 自動戦闘の行動決定ロジックを上書き & 条件判定メソッド追加
    //-----------------------------------------------------------------------------

    const _Game_Actor_makeAutoBattleActions = Game_Actor.prototype.makeAutoBattleActions;
    Game_Actor.prototype.makeAutoBattleActions = function() {
        // actor() がロードされていない場合を考慮 (戦闘テストなどで発生しうる)
        if (!this.actor()) {
            _Game_Actor_makeAutoBattleActions.call(this);
            return;
        }

        const autoBattleActions = this.actor().autoBattleActions;
        if (!autoBattleActions || autoBattleActions.length === 0) {
            _Game_Actor_makeAutoBattleActions.call(this);
            return;
        }

        for (let i = 0; i < this.numActions(); i++) {
            const possibleActions = this.makeActionListFromCustomAI(autoBattleActions);
            let selectedActionData = null;

            if (possibleActions.length > 0) {
                const ratingMax = Math.max(...possibleActions.map(a => a.rating));
                const ratingZero = ratingMax - 3;
                const highPriorityActions = possibleActions.filter(a => a.rating > ratingZero);

                if (highPriorityActions.length > 0) {
                    const sum = highPriorityActions.reduce((r, a) => r + a.rating - ratingZero, 0);
                    let value = Math.randomInt(sum);
                    for (const candidate of highPriorityActions) {
                        value -= candidate.rating - ratingZero;
                        if (value < 0) {
                            selectedActionData = candidate;
                            break;
                        }
                    }
                }
            }

            // Game_Actionオブジェクトを作成
            const gameAction = new Game_Action(this);
            if (selectedActionData) {
                gameAction.setSkill(selectedActionData.skillId);
            } else {
                // 条件に合う行動がない場合は防御
                gameAction.setGuard();
            }
            this.setAction(i, gameAction);
        }
        this.setActionState("waiting");
    };

    Game_Actor.prototype.makeActionListFromCustomAI = function(autoBattleActions) {
        return autoBattleActions.filter(actionData => {
            const skill = $dataSkills[actionData.skillId];
            // スキルが存在し、使用可能で、かつ条件を満たすか
            return skill && this.canUse(skill) && this.meetsActionCondition(actionData);
        });
    };

    Game_Actor.prototype.meetsActionCondition = function(actionData) {
        switch (actionData.conditionType) {
            case 'TURN':
                return this.meetsTurnCondition(actionData.conditionParam1, actionData.conditionParam2);
            case 'HP':
                return this.meetsHpCondition(actionData.conditionParam1, actionData.conditionParam2);
            case 'SWITCH':
                return this.meetsSwitchCondition(actionData.conditionParam1);
            case 'STATE': // [新機能] ステート条件
                return this.meetsStateCondition(actionData.conditionParam1);
            case 'LEVEL': // [新機能] レベル条件
                return this.meetsLevelCondition(actionData.conditionParam1);
            case 'EQUIPW': // [新機能] 武器装備条件
                return this.meetsEquipWCondition(actionData.conditionParam1);
            case 'EQUIPA': // [新機能] 防具装備条件
                return this.meetsEquipACondition(actionData.conditionParam1);
            case 'NONE': // 条件なし
            default:
                return true;
        }
    };

    // --- 条件判定メソッド ---

    Game_Actor.prototype.meetsTurnCondition = function(param1, param2) {
        const n = $gameTroop.turnCount() + 1;
        if (param2 === 0) {
            return n === param1;
        } else {
            return n > 0 && n >= param1 && (n - param1) % param2 === 0;
        }
    };

    Game_Actor.prototype.meetsHpCondition = function(param1, param2) {
        const hpRate = this.hpRate() * 100;
        return hpRate >= param1 && hpRate <= param2;
    };

    Game_Actor.prototype.meetsSwitchCondition = function(param1) {
        return $gameSwitches.value(param1);
    };

    // [新機能] 指定したステートにかかっているか
    Game_Actor.prototype.meetsStateCondition = function(param1) {
        return this.isStateAffected(param1);
    };

    // [新機能] 指定したレベル以上か
    Game_Actor.prototype.meetsLevelCondition = function(param1) {
        return this.level >= param1;
    };

    // [新機能] 指定した武器IDを装備しているか
    Game_Actor.prototype.meetsEquipWCondition = function(param1) {
        // param1 が 0 または NaN の場合は条件を無視（または常にfalseにすることも検討）
        if (!param1) return true;
        return this.weapons().some(weapon => weapon && weapon.id === param1);
    };

    // [新機能] 指定した防具IDを装備しているか
    Game_Actor.prototype.meetsEquipACondition = function(param1) {
        // param1 が 0 または NaN の場合は条件を無視
        if (!param1) return true;
        return this.armors().some(armor => armor && armor.id === param1);
    };

})();